/* 
	PlatoApplication.java

	Title:			Plato
	Author:			David R. Bickel (bickel@mailaps.org)
	Modifications:	11/26/01, printClusteredVectors limited.
					11/20/01, isUsingAbsR added.
	Description:	Plato, an application for cluster analysis using correlation dissimilarities. Created by David R. Bickel on 1/27/01.

Copyright (C) 2001 by David R. Bickel.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

*/

package /*com.DavidBickel.*/Plato;

import java.io.*;
import java.util.*;//for StringTokenizer
import java.lang.*;//for sqrt()


public class Matrix
{
	private String matrixName;
	private float[] [] matrixValues;
	private int numRows;
	private int numColumns=0;
	
	public void clearMatrix()
	{	
		numRows=0;
		numColumns=0;
		matrixValues=null;
	}
	
	public void printMatrix()
	{
		int i;
		int j;
		int numRowsToPrint;
		int numColumnsToPrint;
		int	limit=10;
		
		System.out.println(matrixName+"("+numRows+" rows and "+numColumns+" columns):");
		
		if(numRows<=limit)
			numRowsToPrint=numRows;
		else
			numRowsToPrint=limit;
		if(numColumns<=limit)
			numColumnsToPrint=numColumns;
		else
			numColumnsToPrint=limit;

		
		for(i=0;i<numRowsToPrint;i++)
		{
			for(j=0;j<numColumnsToPrint;j++)
				System.out.print(getElement(i,j)+"\t");
			System.out.println("");
		}
		System.out.println();
	}
	
	public int getNumRows()
	{
		return numRows;
	}
	
	public int getNumColumns()
	{
		return numColumns;
	}
	
	public float[] getRow(int rowNumber)
	{
		return matrixValues[rowNumber];
	}
	
	public float getElement(int rowNumber,int columnNumber)
	{
		return matrixValues[rowNumber][columnNumber];
	}
	
	public String getMatrixName()
	{
		return matrixName;
	}

	private float[] rowOfNumbers(String originalRow) throws Throwable
	{
		int numFloats=0;
		int i=0;
		Float floatObject=new Float(0.);
		float[] floatData;
	
		StringTokenizer st = new StringTokenizer(originalRow);
		while (st.hasMoreTokens()) 
		{
			numFloats++;
			st.nextToken();
		}
		floatData=new float[numFloats];
		if(numColumns==0)
			numColumns=floatData.length;
		if(numColumns!=floatData.length)
			throw new Exception("numColumns:"+numColumns+", but this row has length of "+floatData.length);
		st = new StringTokenizer(originalRow);
		while (st.hasMoreTokens()) 
		{
			floatData[i]=(floatObject.valueOf(st.nextToken())).floatValue();
			i++;
		}
		
		return floatData;
	}

	private void readMatrix() throws Throwable
	{
		BufferedReader in = new BufferedReader(new FileReader(matrixName));
		String lineString = new String();
		float[] lineFloats;
		int i=0;

		System.out.println("Reading "+matrixName);
		while ((lineString=in.readLine())!=null)
		{
			i+=1;
			if(i<=0)
				System.out.println("Row counted: "+lineString);
		}
		numRows=i;
		System.out.println(numRows+" rows");
		matrixValues=new float[numRows] [];
		i=0;
		in = new BufferedReader(new FileReader(matrixName));
		while ((lineString=in.readLine())!=null)
		{
			if(i<=5)
				System.out.println("Row read: "+lineString);
			
			matrixValues[i]=rowOfNumbers(lineString);
			i+=1;
		}
		
		in.close();
	}
	
	
	protected void replaceMatrix(float[] [] newMatrixValues,int newNumColumns,String newMatrixName) throws Throwable
	{	
		int i;
		int	numValuesInRow;
	
		matrixName=newMatrixName;
		matrixValues=newMatrixValues;
		numRows=matrixValues.length;
		numColumns=newNumColumns;
		for(i=0;i<numRows;i++)
			if((numValuesInRow=matrixValues[i].length)>numColumns)
				throw new Exception(numColumns+" is not enough columns for "+matrixName+", row "+i+" of which has "+numValuesInRow+" elements.");
		System.out.println("Matrix initialized from RAM.");
	}
	
	public Matrix()
	{
		matrixName="Matrix to be determined by subclass";
		matrixValues=null;
		numRows=0;
		numColumns=0;
	}
	
	public Matrix(String matrixFileName) throws Throwable
	{
		matrixName=matrixFileName;
		readMatrix();
		System.out.println("Matrix initialized from file");
	}


}


public class MomentStats //several of these methods were copied from DescriptiveStats
{
	
	public float mean(float[] sample)
	{
		int i;
		int sampleSize=sample.length;
		float sum=0;
		
		for (i=0;i<sampleSize;i++)
		{
			sum+=sample[i];
		}
		return sum/sampleSize;
	}

	public float variance(float[] sample)
	{
		float meanValue=mean(sample);
		int i;
		int sampleSize=sample.length;
		float sum=0;
		
		for (i=0;i<sampleSize;i++)
		{
			sum+=(sample[i]-meanValue)*(sample[i]-meanValue);
		}
		return sum/(sampleSize-1);
	}

	public float standardDeviation(float[] sample)
	{
		return (float) Math.sqrt((double) variance(sample));
	}

	public float standardErrorOfSampleMean(float[] sample)
	{
		int sampleSize=sample.length;
	
		return standardDeviation(sample)/((float) Math.sqrt((double) sampleSize));
	}

	public float linearCorrelationCoefficient(float[] sample1,float[] sample2) throws Throwable
	{
		float mean1=mean(sample1);
		float mean2=mean(sample2);
		float value1,value2;
		float sumOfSquares1=0;
		float sumOfSquares2=0;
		float sumOfProducts=0;
		int i;
		int sampleSize=sample1.length;
		
		if(sample1.length!=sample2.length)
			throw new Exception("sample1 has "+sample1.length+", but sample2 has "+sample2.length+" values.");
		for(i=0;i<sampleSize;i++)
		{
			value1=sample1[i]-mean1;
			value2=sample2[i]-mean2;
			sumOfSquares1+=value1*value1;
			sumOfSquares2+=value2*value2;
			sumOfProducts+=value1*value2;
		}
		
		return sumOfProducts/((float) Math.sqrt((double) (sumOfSquares1*sumOfSquares2)));
	}

	public MomentStats()
	{
	
	}

}


public class AssociationMatrix extends Matrix
{
	MomentStats msLocal=new MomentStats();
	int numVectors=0;
	
	public void clearMatrix()
	{
		super.clearMatrix();
		numVectors=0;
	}
	
	public void printMatrix()
	{
		int i;
		int j;
		int numRowsToPrint;
		int numColumnsToPrint;
		int	limit=10;

		System.out.println("numVectors: "+numVectors);
				
		if(numVectors<=limit)
			for(i=0;i<numVectors;i++)
			{
				for(j=0;j<numVectors;j++)
					System.out.print(getElement(i,j)+"\t");
				System.out.println("");
			}
		System.out.println();

	}
	
	
	public float getElement(int vectorIndex1,int vectorIndex2)
	{
		if(vectorIndex2<=vectorIndex1)
			return super.getElement(vectorIndex1,vectorIndex2);
		else
			return super.getElement(vectorIndex2,vectorIndex1);
	}
	
	AssociationMatrix(Matrix vectorsToAssociate) throws Throwable
	{
		int rowNum;
		int colNum;
		int numRows=vectorsToAssociate.getNumRows();
		float [] [] assocMatrixValues;
		
		numVectors=numRows;
		assocMatrixValues=new float[numVectors] [];
		for (rowNum=0;rowNum<numVectors;rowNum++)
		{	
			assocMatrixValues[rowNum]=new float[rowNum+1];
			for (colNum=0;colNum<=rowNum;colNum++)
				assocMatrixValues[rowNum][colNum]=
					msLocal.linearCorrelationCoefficient(vectorsToAssociate.getRow(rowNum),vectorsToAssociate.getRow(colNum));
		}
		replaceMatrix(assocMatrixValues,numVectors,"Association Matrix");
	}	
}






public class PlatoApplication 
{	
	
	boolean isUsingAssociationMatrix;
	MomentStats msPlato=new MomentStats();
	int	maxNumClusters;
	float	alpha;
	boolean isUsingAbsR;
	int maxNumRelocationIterations;
	ClusteredVectors cVectors;
	
	
	class ClusteredVector
	{
		float[] components;
		int dimension;
		int[]	clusterNumbers;//clusterNumber is the index of the centrotype
		boolean[] isCentrotypes;
		float[]	centrotypeAssociations;
		
		
		public void printClusteredVector(int maxNumComponentsToPrint)
		{	
			int i,k;
			
			System.out.println("Dimension: "+dimension);
			for(k=1;k<=maxNumClusters;k++)
			{	
				System.out.println("In cluster #"+getClusterNumber(k)+" of "+k+": "+getIsCentrotype(k)+" centrotype r="+getMeanCentrotypeAssociation(k));
			}
			for(i=0;i<dimension&&i<=maxNumComponentsToPrint;i++)
				System.out.print(getComponent(i)+"\t");
			System.out.println();
			System.out.println();
		}
		
		public float getComponent(int componentIndex)
		{
			return components[componentIndex];
		}
		
		public float[] getComponents()
		{
			return components;
		}
		
		public int getDimension()
		{
			return dimension;
		}
		
		public int getClusterNumber(int numClusters)
		{
			return clusterNumbers[numClusters-1];
		}
		
		public boolean getIsCentrotype(int numClusters)
		{
			return isCentrotypes[numClusters-1];
		}
		
		public float getMeanCentrotypeAssociation(int numClusters)
		{
			return centrotypeAssociations[numClusters-1];
		}
		
		public void setClusterNumber(int numClusters,int clusterNumber)
		{
			clusterNumbers[numClusters-1]=clusterNumber;
		}

		public void setIsCentrotype(int numClusters,boolean isCentrotype)
		{
			isCentrotypes[numClusters-1]=isCentrotype;
		}
		
		public void setCentrotypeAssociation(int numClusters,float centrotypeAssociation)
		{
			centrotypeAssociations[numClusters-1]=centrotypeAssociation;	
		}
	
		ClusteredVector(float[] vectorComponents)
		{
			int i;
			
			components=vectorComponents;
			dimension=components.length;
			clusterNumbers=new int[maxNumClusters];
			isCentrotypes=new boolean[maxNumClusters];
			centrotypeAssociations=new float[maxNumClusters];
			for(i=0;i<maxNumClusters;i++)
			{
				clusterNumbers[i]=-1;
				isCentrotypes[i]=false;
				centrotypeAssociations[i]=((float) 0.);
			}
		}
	}
	
	class ClusteredVectors
	{
		ClusteredVector[]	clusteredVectorArray;
		int	numClusteredVectors;
		AssociationMatrix assocMatrix;
		float[] meanCentrotypeAssociations;
		
		String appendToFileName(String fileName,String whatToAppend)
		{
			String baseName=fileName.substring(0,fileName.length()-4);
			String extension=fileName.substring(fileName.length()-4,fileName.length());
			String newName=baseName.concat(whatToAppend);
			
			return newName.concat(extension);
		}
		
		public void saveClusteredVectors(String baseFileName) throws Throwable
		{
			String clusterNumberFileName=appendToFileName(baseFileName,"-typeNo");
			String isCentrotypeFileName=appendToFileName(baseFileName,"-isType");
			String centrotypeAssociationFileName=appendToFileName(baseFileName,"-typeR");
			String centrotypeDistanceFileName=appendToFileName(baseFileName,"-typeD");
			PrintWriter clusterNumberOut = new PrintWriter(new BufferedWriter(new FileWriter(clusterNumberFileName)));
			PrintWriter isCentrotypeOut = new PrintWriter(new BufferedWriter(new FileWriter(isCentrotypeFileName)));
			PrintWriter centrotypeAssociationOut = new PrintWriter(new BufferedWriter(new FileWriter(centrotypeAssociationFileName)));
			PrintWriter centrotypeDistanceOut = new PrintWriter(new BufferedWriter(new FileWriter(centrotypeDistanceFileName)));
			
			int i,k;
			
			System.out.println("Writing to "+clusterNumberFileName+", "+isCentrotypeFileName+", "+centrotypeDistanceFileName+", and "+centrotypeAssociationFileName);
			for(i=0;i<numClusteredVectors;i++)
			{
				for(k=1;k<=maxNumClusters;k++)
				{
					clusterNumberOut.print((getClusterNumber(i,k)+1)+"\t");
					isCentrotypeOut.print(getIsCentrotype(i,k)+"\t");
					centrotypeAssociationOut.print(getMeanCentrotypeAssociation(i,k)+"\t");
					centrotypeDistanceOut.print(-getMeanCentrotypeAssociation(i,k)+"\t");
				}			
				clusterNumberOut.println();
				isCentrotypeOut.println();
				centrotypeAssociationOut.println();
				centrotypeDistanceOut.println();
			}
			clusterNumberOut.close();			
			isCentrotypeOut.close();			
			centrotypeAssociationOut.close();			
			centrotypeDistanceOut.close();			
		}
		
		
		public float getMeanCentrotypeAssociation(int numClusters)
		{
			return meanCentrotypeAssociations[numClusters-1];
		}
		
		public void printClusteredVectors(int maxNumToPrint)
		{
			int i;
			
			System.out.println();
			for(i=0;i<Math.min(maxNumToPrint,numClusteredVectors);i++)
			{
				System.out.println("Clustered vector number "+i+":");
				clusteredVectorArray[i].printClusteredVector(maxNumToPrint);
			}
			System.out.println();
		}

		public float getComponent(int vectorIndex,int componentIndex)
		{
			return clusteredVectorArray[vectorIndex].getComponent(componentIndex);
		}
		
		public int getDimension(int vectorIndex)
		{
			return clusteredVectorArray[vectorIndex].getDimension();
		}
		
		public int getClusterNumber(int vectorIndex,int numClusters)
		{
			return clusteredVectorArray[vectorIndex].getClusterNumber(numClusters);
		}
		
		public boolean getIsCentrotype(int vectorIndex,int numClusters)
		{
			return clusteredVectorArray[vectorIndex].getIsCentrotype(numClusters);
		}
		
		public float getMeanCentrotypeAssociation(int vectorIndex,int numClusters)
		{
			return clusteredVectorArray[vectorIndex].getMeanCentrotypeAssociation(numClusters);
		}
		
		public void setClusterNumber(int vectorIndex,int numClusters,int clusterNumber)
		{
			clusteredVectorArray[vectorIndex].setClusterNumber(numClusters,clusterNumber);
		}

		public void setIsCentrotype(int vectorIndex,int numClusters,boolean isCentrotype)
		{
			clusteredVectorArray[vectorIndex].setIsCentrotype(numClusters,isCentrotype);
		}
		
		public void setCentrotypeAssociation(int vectorIndex,int numClusters,float centrotypeAssociation)
		{
			clusteredVectorArray[vectorIndex].setCentrotypeAssociation(numClusters,centrotypeAssociation);	
		}
	

		
		float association(int vectorIndex1,int vectorIndex2) throws Throwable
		//this can be implemented whether isUsingAssociationMatrix is true or false
		{
			//float	alpha=((float) 2.);//2.;
		
			if (isUsingAssociationMatrix)
				return assocMatrix.getElement(vectorIndex1,vectorIndex2);
			else
			{
				float	r=msPlato.linearCorrelationCoefficient(clusteredVectorArray[vectorIndex1].getComponents(),clusteredVectorArray[vectorIndex2].getComponents());
				float	similarity=(float) (isUsingAbsR?Math.abs(r):(r+1.)/2.);
				return -((float) Math.sqrt(1.-Math.pow(similarity,alpha)));//negative of Euclidean distance
			};
		}

		float clusterNoncentrotype(int vectorIndex,int numClusters,int[] putativeCentrotypeIndices) throws Throwable
		{
			int k;
			float maxAssoc=-2;
			float assoc;
			int bestCentrotypeIndex=-1;
			int centrotypeIndex;
			
			if(getIsCentrotype(vectorIndex,numClusters))
				throw new Exception("Vector of index "+vectorIndex+" is a centrotype.");
			for(k=1;k<=numClusters;k++)
			{
//System.out.println("vectorIndex=="+vectorIndex+"; k=="+k+" ; putativeCentrotypeIndices.length=="+putativeCentrotypeIndices.length);
				assoc=association(vectorIndex,(centrotypeIndex=putativeCentrotypeIndices[k-1]));
				if(assoc>maxAssoc)
				{
					maxAssoc=assoc;
					bestCentrotypeIndex=centrotypeIndex;
				}
			}
			setClusterNumber(vectorIndex,numClusters,bestCentrotypeIndex);
			setCentrotypeAssociation(vectorIndex,numClusters,maxAssoc);
//System.out.println("clusterNoncentrotype(...) is returning "+maxAssoc);
			return maxAssoc;
		}
		
		void clusterCentrotype(int centrotypeIndex,int numClusters) throws Throwable
		{
			if(!(getIsCentrotype(centrotypeIndex,numClusters)))
				throw new Exception("Vector of index "+centrotypeIndex+" is not a centrotype.");
			setClusterNumber(centrotypeIndex,numClusters,centrotypeIndex);
			setCentrotypeAssociation(centrotypeIndex,numClusters,association(centrotypeIndex,centrotypeIndex));			
		}
		
		float clusterAllVectors(int numClusters,int[] putativeCentrotypeIndices) throws Throwable
		{
			float totalAssoc=0;
			float subTotalAssoc;
			float numAssoc=0;
			int i;
		
//System.out.println(-1);
			if(numClusters!=putativeCentrotypeIndices.length)
				throw new Exception("numClusters=="+numClusters+"; putativeCentrotypeIndices.length=="+putativeCentrotypeIndices.length);
//System.out.println(-numClusters);
			for(i=0;i<numClusteredVectors;i++)
			{
//System.out.println(-i);
				if(getIsCentrotype(i,numClusters))
				{
//System.out.println(-5);
					clusterCentrotype(i,numClusters);
				}
				else
				{
//System.out.println(putativeCentrotypeIndices.length);
					totalAssoc+=clusterNoncentrotype(i,numClusters,putativeCentrotypeIndices);
					numAssoc+=1;
				}
//System.out.println(-6);
			}
//System.out.println(-7);
//System.out.println("clusterAllVectors(...) is returning "+totalAssoc/numAssoc);
			return totalAssoc/numAssoc;
		}
		
		void clusterAllVectors(int numClusters) throws Throwable
		{
			int i,k=0;
			int relocationIteration;
			int centrotypeIndex=-1;
			int	oldCentrotypeIndex;
			float meanAssoc=0;
			float maxMeanAssoc=-2;
			int[] centrotypeIndices=new int[numClusters];

			if(numClusters>1)
				for(i=0;i<numClusteredVectors;i++)
				{	
					if(getIsCentrotype(i,numClusters-1))
					{
						centrotypeIndices[k]=i;
						k++;
						setIsCentrotype(i,numClusters,true);
					}
				}
			if(k!=numClusters-1)
				throw new Exception(k+"==k!=numClusters-1=="+(numClusters-1));
			centrotypeIndices[k]=-1;
			for(relocationIteration=0;relocationIteration<=maxNumRelocationIterations;relocationIteration++)
			{
				for(k=(relocationIteration==0?numClusters-1:0);k<numClusters;k++)
				{	
					oldCentrotypeIndex=centrotypeIndices[k];
					if(oldCentrotypeIndex!=-1)
					{	
						if(!getIsCentrotype(oldCentrotypeIndex,numClusters))
							throw new Exception("getIsCentrotype(centrotypeIndices["+k+"],"+numClusters+") should be true");
						setIsCentrotype(oldCentrotypeIndex,numClusters,false);
					}
					centrotypeIndex=-1;
					for(i=0;i<numClusteredVectors;i++)
					{
						if(!getIsCentrotype(i,numClusters))
						{
							centrotypeIndices[k]=i;
							setIsCentrotype(i,numClusters,true);
							meanAssoc=clusterAllVectors(numClusters,centrotypeIndices);
							setIsCentrotype(i,numClusters,false);
							if(meanAssoc>=maxMeanAssoc)
							{
								maxMeanAssoc=meanAssoc;
								centrotypeIndex=i;
							}
						}
					}
					if(centrotypeIndex==-1)
						throw new Exception("centrotypeIndex=="+centrotypeIndex);
		//System.out.println("Setting isCentrotype of index "+centrotypeIndex+" to true for numClusters=="+numClusters);
					if(getIsCentrotype(centrotypeIndex,numClusters))
						throw new Exception("getIsCentrotype("+centrotypeIndex+","+numClusters+") should be false.");
					setIsCentrotype(centrotypeIndex,numClusters,true);
					centrotypeIndices[k]=centrotypeIndex;
		//System.out.println("numClusters=="+numClusters+"; meanCentrotypeAssociations.length=="+meanCentrotypeAssociations.length);
					meanCentrotypeAssociations[numClusters-1]=clusterAllVectors(numClusters,centrotypeIndices);
					if(getMeanCentrotypeAssociation(numClusters)!=maxMeanAssoc)
					{
						System.out.println();
						System.out.println("WARNING:");
						System.out.println("  getMeanCentrotypeAssociation("+numClusters+") is "+getMeanCentrotypeAssociation(numClusters)+", but maxMeanAssoc is "+maxMeanAssoc+"; meanAssoc=="+meanAssoc+".");
						System.out.println();
					}
				}//end for(k...)
			}//end for(relocationIteration...)
		}
		
		void centrotypeClusterAnalysis(int numClusters) throws Throwable
		{ 
			System.out.println("Finding "+numClusters+" clusters.");
			
			clusterAllVectors(numClusters);
		}
		
		void centrotypeClusterAnalysis() throws Throwable
		{
			int k;
			
			for(k=1;k<=maxNumClusters;k++)
				centrotypeClusterAnalysis(k);
		}

		ClusteredVectors(Matrix listOfVectors) throws Throwable
		{
			int i;
			
			numClusteredVectors=listOfVectors.getNumRows();
			meanCentrotypeAssociations=new float[maxNumClusters];
			clusteredVectorArray=new ClusteredVector[numClusteredVectors];
			for(i=0;i<numClusteredVectors;i++)
				clusteredVectorArray[i]=new ClusteredVector(listOfVectors.getRow(i));
			if(isUsingAssociationMatrix)
			{
				assocMatrix=new AssociationMatrix(listOfVectors);
				assocMatrix.printMatrix();
			}
			centrotypeClusterAnalysis();
		}
	}
	

	public PlatoApplication() throws Throwable 
	{
		System.out.println("\nStandard output from Plato, Version 1.11.\n");

		int i;
		Matrix vectorsToCluster, paramMatrix;
		String inputFileName="VectorRows.txt";
		String paramFileName="Parameters.txt";

		vectorsToCluster=new Matrix(inputFileName);
		vectorsToCluster.printMatrix();
		
		paramMatrix=new Matrix(paramFileName);

		isUsingAssociationMatrix=false;//this works for true; make this depend on catching memory error from AssociationMatrix
		if(paramMatrix.getNumRows()>=1)
		  maxNumClusters=((int) paramMatrix.getElement(0,0));//5;//this works for 100
		else
		  maxNumClusters=2;
		if(paramMatrix.getNumRows()>=2)
		  alpha=((float) paramMatrix.getElement(1,0));
		else
		  alpha=((float) 2.);
		if(paramMatrix.getNumRows()>=3)
		  isUsingAbsR=paramMatrix.getElement(2,0)>=0.;
		else
		  isUsingAbsR=false;
		
		if(maxNumClusters>=vectorsToCluster.getNumRows())
			throw new Exception("maxNumClusters >= # vectors to cluster");
		maxNumRelocationIterations=1;//5;
		
		System.out.println(vectorsToCluster.getMatrixName()+" has "+vectorsToCluster.getNumRows()+" vectors of dimension "+vectorsToCluster.getNumColumns()+".");
		System.out.println("maxNumClusters=="+maxNumClusters+"; stretching exponent alpha=="+alpha+"; is using |correlation|=="+isUsingAbsR);
		
		cVectors=new ClusteredVectors(vectorsToCluster);
//		cVectors.printClusteredVectors(5);
		cVectors.saveClusteredVectors(inputFileName);
		
	}

	// Main entry point
	static public void main(String[] args) throws Throwable
	{
		new PlatoApplication();
		
	}
	
}
